<?php
/**
 * API: Búsqueda de Productos
 * Permite buscar productos por código interno, código de barras, nombre+laboratorio, o laboratorio
 */

header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config/database.php';

try {
    $pdo = getDBConnection();
    
    // Obtener parámetros de búsqueda
    $query = isset($_GET['q']) ? sanitize($_GET['q']) : '';
    
    if (empty($query)) {
        sendJSON(['success' => false, 'error' => 'Parámetro de búsqueda requerido'], 400);
    }
    
    require_once __DIR__ . '/../config/branch.php';
    $sucursal_id = getCurrentBranchId();

    // Búsqueda simple de productos - mostrar TODOS los productos que coincidan
    // LEFT JOIN con detalle_pedidos para ver si ya está pedido en esta sucursal (estado pendiente)
    $sql = "SELECT 
                p.id,
                p.CodigoInterno,
                p.CodigoBarra,
                p.NombreComercial,
                p.NombreGenerico,
                p.laboratorio,
                p.id_presentacion,
                dp.id as detalle_id,
                dp.cantidad_solicitada as pedido_cantidad,
                dp.es_urgente as pedido_urgente,
                dp.existencia_actual as pedido_existencia,
                dp.presentacion_actual as pedido_pres_actual,
                dp.presentacion_solicitada as pedido_pres_solicitada,
                ped.usuario_id as pedido_usuario_id
            FROM productos p
            LEFT JOIN detalle_pedidos dp ON p.id = dp.producto_id 
                AND dp.pedido_id IN (SELECT id FROM pedidos WHERE sucursal_id = ? AND estado = 'pendiente')
            LEFT JOIN pedidos ped ON dp.pedido_id = ped.id
            WHERE p.estado = 1 AND (
                p.CodigoInterno LIKE ? OR
                p.CodigoBarra LIKE ? OR
                p.NombreComercial LIKE ? OR
                p.NombreGenerico LIKE ? OR
                p.laboratorio LIKE ? OR
                CONCAT(p.NombreComercial, ' ', p.laboratorio) LIKE ? OR
                CONCAT(p.NombreGenerico, ' ', p.laboratorio) LIKE ?
            )
            GROUP BY p.id
            ORDER BY p.NombreComercial ASC
            LIMIT 50";
    
    $stmt = $pdo->prepare($sql);
    $searchTerm = "%{$query}%";
    
    // Bind parameters
    $stmt->execute([
        $sucursal_id, // Para el subquery de pedidos
        $searchTerm, // CodigoInterno
        $searchTerm, // CodigoBarra
        $searchTerm, // NombreComercial
        $searchTerm, // NombreGenerico
        $searchTerm, // laboratorio
        $searchTerm, // CONCAT NombreComercial + laboratorio
        $searchTerm  // CONCAT NombreGenerico + laboratorio
    ]);
    
    $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendJSON([
        'success' => true,
        'data' => $productos,
        'count' => count($productos)
    ]);
    
} catch (PDOException $e) {
    error_log("Error en búsqueda de productos: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error al buscar productos: ' . $e->getMessage()], 500);
} catch (Exception $e) {
    error_log("Error general: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error general: ' . $e->getMessage()], 500);
}
