<?php
/**
 * API: Obtener Productos con Paginación
 * Retorna productos con paginación y búsqueda avanzada multi-campo
 */

header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/branch.php';

try {
    $pdo = getDBConnection();
    
    // Parámetros
    $page = isset($_GET['page']) && $_GET['page'] > 0 ? intval($_GET['page']) : 1;
    $limit = isset($_GET['limit']) && $_GET['limit'] > 0 ? intval($_GET['limit']) : 20;
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $sucursal_id = getCurrentBranchId();
    
    $offset = ($page - 1) * $limit;
    
    // Construir condiciones de búsqueda
    $whereConditions = ["p.estado = 1"];
    $params = [];
    
    if (!empty($search)) {
        // Dividir búsqueda en palabras individuales
        $searchWords = preg_split('/\s+/', $search, -1, PREG_SPLIT_NO_EMPTY);
        
        if (count($searchWords) > 0) {
            $searchConditions = [];
            
            foreach ($searchWords as $index => $word) {
                $paramName = "word" . $index;
                $likeValue = "%{$word}%";
                
                // Cada palabra debe aparecer en al menos uno de estos campos
                $searchConditions[] = "(
                    p.CodigoInterno LIKE :{$paramName}_1 OR
                    p.CodigoBarra LIKE :{$paramName}_2 OR
                    p.NombreComercial LIKE :{$paramName}_3 OR
                    p.NombreGenerico LIKE :{$paramName}_4 OR
                    p.laboratorio LIKE :{$paramName}_5 OR
                    CONCAT(p.NombreComercial, ' ', p.laboratorio) LIKE :{$paramName}_6
                )";
                
                // Agregar parámetros para cada campo
                $params["{$paramName}_1"] = $likeValue;
                $params["{$paramName}_2"] = $likeValue;
                $params["{$paramName}_3"] = $likeValue;
                $params["{$paramName}_4"] = $likeValue;
                $params["{$paramName}_5"] = $likeValue;
                $params["{$paramName}_6"] = $likeValue;
            }
            
            // Todas las palabras deben coincidir (AND)
            $whereConditions[] = "(" . implode(" AND ", $searchConditions) . ")";
        }
    }
    
    $whereClause = implode(" AND ", $whereConditions);
    
    // Contar total de productos
    $countSql = "SELECT COUNT(DISTINCT p.id) as total 
                 FROM productos p 
                 WHERE {$whereClause}";
    
    $countStmt = $pdo->prepare($countSql);
    foreach ($params as $key => $value) {
        $countStmt->bindValue(":{$key}", $value, PDO::PARAM_STR);
    }
    $countStmt->execute();
    $totalProducts = $countStmt->fetch()['total'];
    $totalPages = ceil($totalProducts / $limit);
    
    // Obtener productos de la página actual
    // LEFT JOIN con detalle_pedidos para ver si ya está pedido en esta sucursal (estado pendiente)
    $sql = "SELECT DISTINCT
                p.id,
                p.CodigoInterno,
                p.CodigoBarra,
                p.NombreComercial,
                p.NombreGenerico,
                p.laboratorio,
                p.id_presentacion,
                dp.id as detalle_id,
                dp.cantidad_solicitada as pedido_cantidad,
                dp.es_urgente as pedido_urgente,
                dp.existencia_actual as pedido_existencia,
                dp.presentacion_actual as pedido_pres_actual,
                dp.presentacion_solicitada as pedido_pres_solicitada,
                ped.usuario_id as pedido_usuario_id
            FROM productos p
            LEFT JOIN detalle_pedidos dp ON p.id = dp.producto_id 
                AND dp.pedido_id IN (SELECT id FROM pedidos WHERE sucursal_id = :sucursal_id AND estado = 'pendiente')
            LEFT JOIN pedidos ped ON dp.pedido_id = ped.id
            WHERE {$whereClause}
            GROUP BY p.id
            ORDER BY p.NombreComercial ASC
            LIMIT :limit OFFSET :offset";
    
    $stmt = $pdo->prepare($sql);
    
    // Bind search params
    foreach ($params as $key => $value) {
        $stmt->bindValue(":{$key}", $value, PDO::PARAM_STR);
    }
    
    // Bind pagination and branch params
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->bindValue(':sucursal_id', $sucursal_id, PDO::PARAM_INT);
    
    $stmt->execute();
    
    $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    sendJSON([
        'success' => true,
        'data' => $products,
        'pagination' => [
            'current_page' => $page,
            'total_pages' => $totalPages,
            'total_products' => (int)$totalProducts,
            'per_page' => $limit,
            'has_next' => $page < $totalPages,
            'has_prev' => $page > 1,
            'showing_from' => $offset + 1,
            'showing_to' => min($offset + $limit, $totalProducts)
        ]
    ]);
    
} catch (PDOException $e) {
    error_log("Error al obtener productos: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error al obtener productos: ' . $e->getMessage()], 500);
} catch (Exception $e) {
    error_log("Error general: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error general: ' . $e->getMessage()], 500);
}
